<?php
/**
 * @brief		Background Task: Resychronise the first post of a topic generated by Pages
 * @author		<a href='https://www.invisioncommunity.com'>Invision Power Services, Inc.</a>
 * @copyright	(c) Invision Power Services, Inc.
 * @license		https://www.invisioncommunity.com/legal/standards/
 * @package		Invision Community
 * @subpackage	Content
 * @since		7 Oct 2016
 */

namespace IPS\cms\extensions\core\Queue;

/* To prevent PHP errors (extending class does not exist) revealing path */

use Exception;
use IPS\Application;
use IPS\cms\Databases;
use IPS\cms\Records;
use IPS\Db;
use IPS\Extensions\QueueAbstract;
use IPS\Member;
use OutOfRangeException;
use function count;
use function defined;
use const IPS\REBUILD_NORMAL;

if ( !defined( '\IPS\SUITE_UNIQUE_KEY' ) )
{
	header( ( $_SERVER['SERVER_PROTOCOL'] ?? 'HTTP/1.0' ) . ' 403 Forbidden' );
	exit;
}

/**
 * Background Task: Rebuild database records
 */
class ResyncTopicContent extends QueueAbstract
{
	/**
	 * @brief Number of content items to rebuild per cycle
	 */
	public int $rebuild	= REBUILD_NORMAL;

	/**
	 * Parse data before queuing
	 *
	 * @param	array	$data
	 * @return	array|null
	 */
	public function preQueueData( array $data ): ?array
	{
		try
		{
			$database = Databases::load( $data['databaseId'] );
			
			if ( $database->forum_record )
			{
				$data['count'] = (int) Db::i()->select( 'COUNT(*)', 'cms_custom_database_' . $data['databaseId'] )->first();
			}
			else
			{
				$cats = iterator_to_array( Db::i()->select( 'category_id', 'cms_database_categories', array( 'category_forum_record=1 AND category_database_id=?', $data['databaseId'] ) ) );
				
				if ( count( $cats ) )
				{
					$data['count'] = (int) Db::i()->select( 'COUNT(*)', 'cms_custom_database_' . $data['databaseId'], array( Db::i()->in( 'category_id', $cats ) ) )->first();
				}
				else
				{
					return null;
				}
			}
		}
		catch( Exception $ex )
		{
			throw new OutOfRangeException;
		}
		
		if( $data['count'] == 0 )
		{
			return null;
		}

		$data['completed'] = 0;
		
		return $data;
	}

	/**
	 * Run Background Task
	 *
	 * @param	mixed						$data	Data as it was passed to \IPS\Task::queue()
	 * @param	int							$offset	Offset
	 * @return	int							New offset
	 * @throws	\IPS\Task\Queue\OutOfRangeException	Indicates offset doesn't exist and thus task is complete
	 */
	public function run( mixed &$data, int $offset ): int
	{
		/* @var Records $class */
		$class	= '\IPS\cms\Records' . $data['databaseId'];
		$where	= array( 'primary_id_field>?', $offset );
		$last	= NULL;

		try
		{
			$database = Databases::load( $data['databaseId'] );
		}
		catch ( OutOfRangeException $e )
		{
			throw new \IPS\Task\Queue\OutOfRangeException;
		}

		if ( ! $database->forum_record )
		{
			$cats = iterator_to_array( Db::i()->select( 'category_id', 'cms_database_categories', array( 'category_forum_record=1 AND category_database_id=?', $data['databaseId'] ) ) );
			$where[] = array( Db::i()->in( 'category_id', $cats ) );
		}

		if ( Db::i()->checkForTable( 'cms_custom_database_' . $data['databaseId'] ) )
		{
			foreach ( Db::i()->select( '*', 'cms_custom_database_' . $data['databaseId'], $where, 'primary_id_field asc', array( 0, $this->rebuild ) ) as $row )
			{
				try
				{
					$record = $class::constructFromData( $row );
					$record->syncTopic();
				}
				catch( Exception $ex ) { }
				
				$data['completed']++;
				$last = $row['primary_id_field'];
			}
		}
		
		if( $last === NULL )
		{
			throw new \IPS\Task\Queue\OutOfRangeException;
		}

		return $last;
	}
	
	/**
	 * Get Progress
	 *
	 * @param	mixed					$data	Data as it was passed to \IPS\Task::queue()
	 * @param	int						$offset	Offset
	 * @return	array( 'text' => 'Doing something...', 'complete' => 50 )	Text explaining task and percentage complete
	 * @throws	OutOfRangeException	Indicates offset doesn't exist and thus task is complete
	 */
	public function getProgress( mixed $data, int $offset ): array
	{
		$title = ( Application::appIsEnabled('cms') ) ? Databases::load( $data['databaseId'] )->_title : 'Database #' . $data['databaseId'];
		return array( 'text' => Member::loggedIn()->language()->addToStack('rebuilding_cms_database_sync_topics', FALSE, array( 'sprintf' => array( $title ) ) ), 'complete' => $data['count'] ? ( round( 100 / $data['count'] * $data['completed'], 2 ) ) : 100 );
	}	
}