<?php

namespace XF\Entity;

use XF\Api\Result\EntityResult;
use XF\BbCode\RenderableContentInterface;
use XF\Mvc\Entity\AbstractCollection;
use XF\Mvc\Entity\Entity;
use XF\Mvc\Entity\Structure;
use XF\Repository\AttachmentRepository;
use XF\Repository\UserAlertRepository;
use XF\Spam\ContentChecker;

/**
 * COLUMNS
 * @property int|null $profile_post_comment_id
 * @property int $profile_post_id
 * @property int $user_id
 * @property string $username
 * @property int $comment_date
 * @property string $message
 * @property int $ip_id
 * @property string $message_state
 * @property int $attach_count
 * @property int $warning_id
 * @property string $warning_message
 * @property array|null|null $embed_metadata
 * @property int $reaction_score
 * @property array|null $reactions_
 * @property array|null $reaction_users_
 *
 * GETTERS
 * @property-read mixed $Unfurls
 * @property mixed $reactions
 * @property mixed $reaction_users
 * @property-read array $Embeds
 *
 * RELATIONS
 * @property-read User|null $User
 * @property-read ProfilePost|null $ProfilePost
 * @property-read DeletionLog|null $DeletionLog
 * @property-read ApprovalQueue|null $ApprovalQueue
 * @property-read AbstractCollection<Attachment> $Attachments
 * @property-read AbstractCollection<ReactionContent> $Reactions
 */
class ProfilePostComment extends Entity implements RenderableContentInterface, LinkableInterface, ViewableInterface
{
	use EmbedRendererTrait;
	use ReactionTrait;

	public function canView(&$error = null)
	{
		$visitor = \XF::visitor();

		/** @var ProfilePost $profilePost */
		$profilePost = $this->ProfilePost;
		if (!$profilePost)
		{
			return false;
		}

		if ($this->message_state == 'moderated')
		{
			if (
				!$profilePost->canViewModeratedComments()
				&& (!$visitor->user_id || $visitor->user_id != $this->user_id)
			)
			{
				$error = \XF::phraseDeferred('requested_comment_not_found');
				return false;
			}
		}
		else if ($this->message_state == 'deleted')
		{
			if (!$profilePost->canViewDeletedComments())
			{
				$error = \XF::phraseDeferred('requested_comment_not_found');
				return false;
			}
		}

		return $profilePost->canView($error);
	}

	public function canViewAttachments(&$error = null): bool
	{
		$profilePost = $this->ProfilePost;
		if (!$profilePost)
		{
			return false;
		}

		return $profilePost->canViewAttachments($error);
	}

	public function canEdit(&$error = null)
	{
		$visitor = \XF::visitor();

		if (!$visitor->user_id)
		{
			return false;
		}

		if ($visitor->user_id == $this->user_id)
		{
			return $visitor->hasPermission('profilePost', 'editOwn');
		}
		else
		{
			return $visitor->hasPermission('profilePost', 'editAny');
		}
	}

	public function canDelete($type = 'soft', &$error = null)
	{
		$visitor = \XF::visitor();
		if (!$visitor->user_id)
		{
			return false;
		}

		if ($type != 'soft' && !$visitor->hasPermission('profilePost', 'hardDeleteAny'))
		{
			return false;
		}

		if ($visitor->hasPermission('profilePost', 'deleteAny'))
		{
			return true;
		}

		return (
			(
				$this->ProfilePost
				&& $visitor->user_id == $this->ProfilePost->profile_user_id
				&& $visitor->hasPermission('profilePost', 'manageOwn')
			)
			||
			(
				$visitor->user_id == $this->user_id
				&& $visitor->hasPermission('profilePost', 'deleteOwn')
			)
		);
	}

	public function canUndelete(&$error = null)
	{
		$visitor = \XF::visitor();
		return ($visitor->user_id && $visitor->hasPermission('profilePost', 'undelete'));
	}

	public function canApproveUnapprove(&$error = null)
	{
		$visitor = \XF::visitor();
		return ($visitor->user_id && $visitor->hasPermission('profilePost', 'approveUnapprove'));
	}

	public function canWarn(&$error = null)
	{
		$visitor = \XF::visitor();

		if (!$this->user_id
			|| !$visitor->user_id
			|| $this->user_id == $visitor->user_id
			|| !$visitor->hasPermission('profilePost', 'warn')
		)
		{
			return false;
		}

		if ($this->warning_id)
		{
			$error = \XF::phraseDeferred('user_has_already_been_warned_for_this_content');
			return false;
		}

		return ($this->User && $this->User->isWarnable());
	}

	public function canReport(&$error = null, ?User $asUser = null)
	{
		$asUser = $asUser ?: \XF::visitor();
		return $asUser->canReport($error);
	}

	public function canReact(&$error = null)
	{
		$visitor = \XF::visitor();
		if (!$visitor->user_id)
		{
			return false;
		}

		if ($this->message_state != 'visible')
		{
			return false;
		}

		if ($this->user_id == $visitor->user_id)
		{
			$error = \XF::phraseDeferred('reacting_to_your_own_content_is_considered_cheating');
			return false;
		}

		return $visitor->hasPermission('profilePost', 'react');
	}

	public function canSendModeratorActionAlert()
	{
		$visitor = \XF::visitor();

		if (!$visitor->user_id || $visitor->user_id == $this->user_id)
		{
			return false;
		}

		if ($this->message_state != 'visible')
		{
			return false;
		}

		return (
			$this->ProfilePost->canSendModeratorActionAlert()
			&& $this->message_state == 'visible'
		);
	}

	public function isVisible()
	{
		return (
			$this->message_state == 'visible'
			&& $this->ProfilePost
			&& $this->ProfilePost->message_state == 'visible'
		);
	}

	public function isAttachmentEmbedded($attachmentId)
	{
		if (!$this->embed_metadata)
		{
			return false;
		}

		if ($attachmentId instanceof Attachment)
		{
			$attachmentId = $attachmentId->attachment_id;
		}

		return isset($this->embed_metadata['attachments'][$attachmentId]);
	}

	public function isIgnored()
	{
		return \XF::visitor()->isIgnoring($this->user_id);
	}

	public function isLastComment()
	{
		return (
			$this->ProfilePost
			&& $this->ProfilePost->last_comment_date == $this->comment_date
		);
	}

	public function canCleanSpam()
	{
		return (\XF::visitor()->canCleanSpam() && $this->User && $this->User->isPossibleSpammer());
	}

	public function getBbCodeRenderOptions($context, $type)
	{
		$renderOptions = [
			'entity' => $this,
			'user' => $this->User,
			'treatAsStructuredText' => true,
			'attachments' => $this->attach_count ? $this->Attachments : [],
			'viewAttachments' => $this->canViewAttachments(),
			'unfurls' => $this->Unfurls ?: [],
			'images' => $this->embed_metadata['images'] ?? [],
		];

		$this->addEmbedRendererBbCodeOptions($renderOptions, $context, $type);

		return $renderOptions;
	}

	public function getUnfurls()
	{
		return $this->_getterCache['Unfurls'] ?? [];
	}

	public function setUnfurls($unfurls)
	{
		$this->_getterCache['Unfurls'] = $unfurls;
	}

	protected function _postSave()
	{
		$visibilityChange = $this->isStateChanged('message_state', 'visible');
		$approvalChange = $this->isStateChanged('message_state', 'moderated');
		$deletionChange = $this->isStateChanged('message_state', 'deleted');

		if ($this->isUpdate())
		{
			if ($visibilityChange == 'enter')
			{
				$this->commentMadeVisible();

				if ($approvalChange)
				{
					$this->submitHamData();
				}
			}
			else if ($visibilityChange == 'leave')
			{
				$this->commentHidden();
			}

			if ($deletionChange == 'leave' && $this->DeletionLog)
			{
				$this->DeletionLog->delete();
			}

			if ($approvalChange == 'leave' && $this->ApprovalQueue)
			{
				$this->ApprovalQueue->delete();
			}
		}

		if ($approvalChange == 'enter')
		{
			$approvalQueue = $this->getRelationOrDefault('ApprovalQueue', false);
			$approvalQueue->content_date = $this->comment_date;
			$approvalQueue->save();
		}
		else if ($deletionChange == 'enter' && !$this->DeletionLog)
		{
			$delLog = $this->getRelationOrDefault('DeletionLog', false);
			$delLog->setFromVisitor();
			$delLog->save();
		}

		$this->updateProfilePostRecord();

		if ($this->isUpdate() && $this->getOption('log_moderator'))
		{
			$this->app()->logger()->logModeratorChanges('profile_post_comment', $this);
		}
	}

	protected function commentMadeVisible()
	{
	}

	protected function commentHidden($hardDelete = false)
	{
		$alertRepo = $this->repository(UserAlertRepository::class);
		$alertRepo->fastDeleteAlertsForContent('profile_post_comment', $this->profile_post_comment_id);
	}

	protected function updateProfilePostRecord()
	{
		if (!$this->ProfilePost || !$this->ProfilePost->exists())
		{
			return;
		}

		$visibilityChange = $this->isStateChanged('message_state', 'visible');
		if ($visibilityChange == 'enter')
		{
			$this->ProfilePost->commentAdded($this);
			$this->ProfilePost->save();
		}
		else if ($visibilityChange == 'leave')
		{
			$this->ProfilePost->commentRemoved($this);
			$this->ProfilePost->save();
		}
		else if ($this->isInsert())
		{
			$this->ProfilePost->rebuildCommentIds();
			$this->ProfilePost->save();
		}
	}

	protected function submitHamData()
	{
		/** @var ContentChecker $submitter */
		$submitter = $this->app()->container('spam.contentHamSubmitter');
		$submitter->submitHam('profile_post_comment', $this->profile_post_comment_id);
	}

	protected function _postDelete()
	{
		if ($this->message_state == 'visible')
		{
			$this->commentHidden(true);
		}

		if ($this->ProfilePost && $this->message_state == 'visible')
		{
			$this->ProfilePost->commentRemoved($this);
			$this->ProfilePost->save();
		}

		if ($this->message_state == 'deleted' && $this->DeletionLog)
		{
			$this->DeletionLog->delete();
		}

		if ($this->message_state == 'moderated' && $this->ApprovalQueue)
		{
			$this->ApprovalQueue->delete();
		}

		if ($this->getOption('log_moderator'))
		{
			$this->app()->logger()->logModeratorAction('profile_post_comment', $this, 'delete_hard');
		}

		$attachRepo = $this->repository(AttachmentRepository::class);
		$attachRepo->fastDeleteContentAttachments('profile_post_comment', $this->profile_post_comment_id);
	}

	public function softDelete($reason = '', ?User $byUser = null)
	{
		$byUser = $byUser ?: \XF::visitor();

		if ($this->message_state == 'deleted')
		{
			return false;
		}

		$this->message_state = 'deleted';

		/** @var DeletionLog $deletionLog */
		$deletionLog = $this->getRelationOrDefault('DeletionLog');
		$deletionLog->setFromUser($byUser);
		$deletionLog->delete_reason = $reason;

		$this->save();

		return true;
	}

	/**
	 * @param EntityResult $result
	 * @param int $verbosity
	 * @param array $options
	 *
	 * @api-out str $username
	 * @api-out str $message_parsed HTML parsed version of the message contents.
	 * @api-out bool $can_edit
	 * @api-out bool $can_soft_delete
	 * @api-out bool $can_hard_delete
	 * @api-out bool $can_react
	 * @api-out bool $can_view_attachments
	 * @api-out Attachment[] $Attachments <cond> Attachments to this profile post, if it has any.
	 * @api-out ProfilePost $ProfilePost <cond> If requested by context, the profile post this comment relates to.
	 *
	 * @api-see XF\Entity\ReactionTrait::addReactionStateToApiResult
	 */
	protected function setupApiResultData(
		EntityResult $result,
		$verbosity = self::VERBOSITY_NORMAL,
		array $options = []
	)
	{
		$result->username = $this->User ? $this->User->username : $this->username;

		if (!empty($options['with_post']))
		{
			$result->includeRelation('ProfilePost', self::VERBOSITY_NORMAL, [
				'with_profile' => true,
			]);
		}

		if ($this->attach_count)
		{
			// note that we allow viewing of thumbs and metadata, regardless of permissions, when viewing the
			// content an attachment is connected to
			$result->includeRelation('Attachments');
		}

		$result->message_parsed = $this->app()->bbCode()->render($this->message, 'apiHtml', 'profile_post_comment:api', $this);

		$this->addReactionStateToApiResult($result);

		$result->view_url = $this->getContentUrl(true);

		if ($result->getResultType() === EntityResult::TYPE_API)
		{
			$result->can_edit = $this->canEdit();
			$result->can_soft_delete = $this->canDelete();
			$result->can_hard_delete = $this->canDelete('hard');
			$result->can_react = $this->canReact();
			$result->can_view_attachments = $this->canViewAttachments();
		}
	}

	public function getContentUrl(bool $canonical = false, array $extraParams = [], $hash = null)
	{
		$route = $canonical ? 'canonical:profile-posts/comments' : 'profile-posts/comments';
		return $this->app()->router('public')->buildLink($route, $this, $extraParams, $hash);
	}

	public function getContentPublicRoute()
	{
		return 'profile-posts/comments';
	}

	public function getContentTitle(string $context = '')
	{
		return \XF::phrase('profile_post_comment_by_x', ['username' => $this->username]);
	}

	public static function getStructure(Structure $structure)
	{
		$structure->table = 'xf_profile_post_comment';
		$structure->shortName = 'XF:ProfilePostComment';
		$structure->contentType = 'profile_post_comment';
		$structure->primaryKey = 'profile_post_comment_id';
		$structure->columns = [
			'profile_post_comment_id' => ['type' => self::UINT, 'autoIncrement' => true, 'nullable' => true],
			'profile_post_id' => ['type' => self::UINT, 'required' => true, 'api' => true],
			'user_id' => ['type' => self::UINT, 'required' => true, 'api' => true],
			'username' => ['type' => self::STR, 'maxLength' => 50,
				'required' => 'please_enter_valid_name',
			],
			'comment_date' => ['type' => self::UINT, 'required' => true, 'default' => \XF::$time, 'api' => true],
			'message' => ['type' => self::STR,
				'required' => 'please_enter_valid_message', 'api' => true,
			],
			'ip_id' => ['type' => self::UINT, 'default' => 0],
			'message_state' => ['type' => self::STR, 'default' => 'visible',
				'allowedValues' => ['visible', 'moderated', 'deleted'], 'api' => true,
			],
			'attach_count' => ['type' => self::UINT, 'max' => 65535, 'forced' => true, 'default' => 0],
			'warning_id' => ['type' => self::UINT, 'default' => 0],
			'warning_message' => ['type' => self::STR, 'default' => '', 'maxLength' => 255, 'api' => true],
			'embed_metadata' => ['type' => self::JSON_ARRAY, 'nullable' => true, 'default' => null],
		];
		$structure->behaviors = [
			'XF:Reactable' => ['stateField' => 'message_state'],
			'XF:Indexable' => [
				'checkForUpdates' => ['message', 'user_id', 'comment_date', 'message_state'],
				'enqueueIndexNow' => true,
			],
			'XF:NewsFeedPublishable' => [
				'usernameField' => 'username',
				'dateField' => 'comment_date',
			],
			'XF:Webhook' => [],
		];
		$structure->getters = [
			'Unfurls' => true,
		];
		$structure->relations = [
			'User' => [
				'entity' => 'XF:User',
				'type' => self::TO_ONE,
				'conditions' => 'user_id',
				'primary' => true,
				'api' => true,
			],
			'ProfilePost' => [
				'entity' => 'XF:ProfilePost',
				'type' => self::TO_ONE,
				'conditions' => 'profile_post_id',
				'primary' => true,
			],
			'DeletionLog' => [
				'entity' => 'XF:DeletionLog',
				'type' => self::TO_ONE,
				'conditions' => [
					['content_type', '=', 'profile_post_comment'],
					['content_id', '=', '$profile_post_comment_id'],
				],
				'primary' => true,
			],
			'ApprovalQueue' => [
				'entity' => 'XF:ApprovalQueue',
				'type' => self::TO_ONE,
				'conditions' => [
					['content_type', '=', 'profile_post_comment'],
					['content_id', '=', '$profile_post_comment_id'],
				],
				'primary' => true,
			],
			'Attachments' => [
				'entity' => 'XF:Attachment',
				'type' => self::TO_MANY,
				'conditions' => [
					['content_type', '=', 'profile_post_comment'],
					['content_id', '=', '$profile_post_comment_id'],
				],
				'with' => 'Data',
				'order' => 'attach_date',
			],
		];
		$structure->options = [
			'log_moderator' => true,
		];
		$structure->defaultWith = ['ProfilePost'];

		$structure->withAliases = [
			'full' => [
				'User',
				function ()
				{
					$userId = \XF::visitor()->user_id;
					if ($userId)
					{
						return 'Reactions|' . $userId;
					}

					return null;
				},
			],
			'api' => [
				'User',
				'User.api',
				function ($withParams)
				{
					if (!empty($withParams['post']))
					{
						return ['ProfilePost.api|profile'];
					}
				},
			],
		];

		static::addReactableStructureElements($structure);
		static::addEmbedRendererStructureElements($structure);

		return $structure;
	}
}
