<?php

namespace XF\Entity;

use XF\Criteria\UserCriteria;
use XF\Mvc\Entity\Entity;
use XF\Mvc\Entity\Structure;
use XF\Phrase;

/**
 * COLUMNS
 * @property int|null $trophy_id
 * @property int $trophy_points
 * @property array|null $user_criteria
 *
 * GETTERS
 * @property-read Phrase $title
 * @property-read Phrase $description
 *
 * RELATIONS
 * @property-read \XF\Entity\Phrase|null $MasterTitle
 * @property-read \XF\Entity\Phrase|null $MasterDescription
 */
class Trophy extends Entity
{
	/**
	 * @return Phrase
	 */
	public function getTitle()
	{
		return \XF::phrase($this->getPhraseName(true));
	}

	/**
	 * @return Phrase
	 */
	public function getDescription()
	{
		return \XF::phrase($this->getPhraseName(false));
	}

	public function getPhraseName($title)
	{
		return 'trophy_' . ($title ? 'title' : 'description') . '.' . $this->trophy_id;
	}

	public function getMasterPhrase($title)
	{
		$phrase = $title ? $this->MasterTitle : $this->MasterDescription;
		if (!$phrase)
		{
			$phrase = $this->_em->create(\XF\Entity\Phrase::class);
			$phrase->title = $this->_getDeferredValue(function () use ($title) { return $this->getPhraseName($title); }, 'save');
			$phrase->language_id = 0;
			$phrase->addon_id = '';
		}

		return $phrase;
	}

	protected function verifyUserCriteria(&$criteria)
	{
		$userCriteria = $this->app()->criteria(UserCriteria::class, $criteria);
		$criteria = $userCriteria->getCriteria();
		return true;
	}

	protected function _postSave()
	{
		if ($this->isUpdate() && $this->isChanged('trophy_points'))
		{
			$this->updateTrophyPoints($this->trophy_id, $this->getExistingValue('trophy_points'), $this->trophy_points);
		}
	}

	protected function _postDelete()
	{
		if ($this->MasterTitle)
		{
			$this->MasterTitle->delete();
		}
		if ($this->MasterDescription)
		{
			$this->MasterDescription->delete();
		}

		$this->updateTrophyPoints($this->trophy_id, $this->trophy_points, 0);

		$this->db()->delete('xf_user_trophy', 'trophy_id = ?', $this->trophy_id);

		$this->db()->delete(
			'xf_user_alert',
			"content_type = 'trophy' AND content_id = ? AND action = 'award'",
			[$this->trophy_id]
		);
	}

	protected function updateTrophyPoints($trophyId, $oldPoints, $newPoints)
	{
		$adjust = $oldPoints - $newPoints;

		$this->db()->query('
			UPDATE xf_user SET
				trophy_points = IF(trophy_points > ?, trophy_points - ?, 0)
			WHERE user_id IN (
				SELECT user_id
				FROM xf_user_trophy
				WHERE trophy_id = ?
			)
		', [$adjust, $adjust, $trophyId]);
	}

	public static function getStructure(Structure $structure)
	{
		$structure->table = 'xf_trophy';
		$structure->shortName = 'XF:Trophy';
		$structure->primaryKey = 'trophy_id';
		$structure->columns = [
			'trophy_id' => ['type' => self::UINT, 'autoIncrement' => true, 'nullable' => true],
			'trophy_points' => ['type' => self::UINT, 'required' => true],
			'user_criteria' => ['type' => self::JSON_ARRAY, 'default' => [],
				'required' => 'please_select_criteria_that_must_be_met',
			],
		];
		$structure->getters = [
			'title' => true,
			'description' => true,
		];
		$structure->relations = [
			'MasterTitle' => [
				'entity' => 'XF:Phrase',
				'type' => self::TO_ONE,
				'conditions' => [
					['language_id', '=', 0],
					['title', '=', 'trophy_title.', '$trophy_id'],
				],
			],
			'MasterDescription' => [
				'entity' => 'XF:Phrase',
				'type' => self::TO_ONE,
				'conditions' => [
					['language_id', '=', 0],
					['title', '=', 'trophy_description.', '$trophy_id'],
				],
			],
		];

		return $structure;
	}
}
